Create or Replace PACKAGE pkg_alkindi_evaluation AS

  TYPE evaluation_cursor_type is ref cursor;

  PROCEDURE sp_insert_evals (
    i_user_id             in EVALUATION.user_id%type,
    i_product_id          in EVALUATION.product_id%type,
    i_evaluation_scale_id in EVALUATION.evaluation_scale_id%type
  );

  PROCEDURE sp_sel_prodeval_by_user (
    i_user_id           in  EVALUATION.user_id%type,
    o_evaluation_cursor out evaluation_cursor_type,
    o_error_code        out number
  );

  PROCEDURE sp_sel_user_data (
    i_user_id           in  USER_ID.user_id%type,
    o_evaluation_cursor out evaluation_cursor_type,
    o_error_code        out number
  );

  PROCEDURE sp_sel_new_prod_data_by_pc (
    i_product_cluster_id in  PRODUCT_CLUSTER.product_cluster_id%type,
    i_user_id            in  USER_ID.user_id%type,
    o_evaluation_cursor  out evaluation_cursor_type,
    o_error_code         out number
  );

  PROCEDURE sp_sel_new_prod_data_by_pc1 (
    i_product_cluster_id in  PRODUCT_CLUSTER.product_cluster_id%type,
    i_r                  in  PROD_STAT.frn_user_seen%type,
    i_q                  in  PROD_STAT.tot_user_eval%type,
    i_user_id            in  USER_ID.user_id%type,
    o_evaluation_cursor  out evaluation_cursor_type,
    o_error_code         out number
  );

  PROCEDURE sp_sel_new_prod_data_by_pc2 (
    i_product_cluster_id in  PRODUCT_CLUSTER.product_cluster_id%type,
    i_r                  in  PROD_STAT.frn_user_seen%type,
    i_q                  in  PROD_STAT.tot_user_eval%type,
    i_user_id            in  USER_ID.user_id%type,
    o_evaluation_cursor  out evaluation_cursor_type,
    o_error_code         out number
  );

  PROCEDURE sp_sel_new_prod_data_by_user (
    i_user_id           in  USER_ID.user_id%type,
    o_evaluation_cursor out evaluation_cursor_type,
    o_error_code        out number
  );

  PROCEDURE sp_sel_old_prod_data_by_pc (
    i_product_cluster_id in  PRODUCT_CLUSTER.product_cluster_id%type,
    i_user_id            in  USER_ID.user_id%type,
    o_evaluation_cursor  out evaluation_cursor_type,
    o_error_code         out number
  );

  PROCEDURE sp_sel_old_prod_data_by_pc1 (
    i_product_cluster_id in  PRODUCT_CLUSTER.product_cluster_id%type,
    i_r                  in  PROD_STAT.frn_user_seen%type,
    i_q                  in  PROD_STAT.tot_user_eval%type,
    i_omega              in  PROD_STAT.tot_user_seen%type,
    i_delta              in  PROD_STAT.tot_user_seen%type,
    i_user_id            in  USER_ID.user_id%type,
    o_evaluation_cursor  out evaluation_cursor_type,
    o_error_code         out number
  );

  PROCEDURE sp_sel_old_prod_data_by_pc2 (
    i_product_cluster_id in  PRODUCT_CLUSTER.product_cluster_id%type,
    i_r                  in  PROD_STAT.frn_user_seen%type,
    i_q                  in  PROD_STAT.tot_user_eval%type,
    i_omega              in  PROD_STAT.tot_user_seen%type,
    i_delta              in  PROD_STAT.tot_user_seen%type,
    i_user_id            in  USER_ID.user_id%type,
    o_evaluation_cursor  out evaluation_cursor_type,
    o_error_code         out number
  );

  PROCEDURE sp_sel_old_prod_data_by_user (
    i_user_id           in  USER_ID.user_id%type,
    o_evaluation_cursor	out evaluation_cursor_type,
    o_error_code        out number
  );

  PROCEDURE sp_sel_prod_by_user_data (
    i_product_cluster_id in  PROD_UC_STAT.product_cluster_id%type,
    i_user_id            in  USER_ID.user_id%type,
    o_evaluation_cursor  out evaluation_cursor_type,
    o_error_code         out number
  );

  PROCEDURE sp_cnt_prod_seen_by_user (
    i_user_id       in  evaluation.user_id%type,
    o_tot_prod_seen out number,
    o_error_code    out number
  );

  PROCEDURE getUserStats (
    i_user_id       in  evaluation.user_id%type,
    o_tot_prod_eval out number,
    o_tot_prod_seen out number,
    o_frn_prod_seen out number,
    o_tot_core_prod_eval out number,
    o_tot_selectable_prod_eval out number,
    o_error_code    out number
  );

  PROCEDURE getEveryoneStats (
    o_tot_prod_eval out number,
    o_tot_prod_seen out number,
    o_frn_prod_seen out number,
    o_tot_core_prods out number,
    o_tot_selectable_prods out number,
    o_error_code    out number
  );

END pkg_alkindi_evaluation;
/

Create or Replace PACKAGE BODY pkg_alkindi_evaluation AS

-- ==========================================================================================
-- Insert evaluations into the table EVALUATION. When there are duplicate values on index,
-- then insert the evaluations into EVALUATION_HISTORY.
-- Simultaneously, the EVALUATION table is updated by the new records.

PROCEDURE sp_insert_evals (
  i_user_id             in EVALUATION.user_id%type,
  i_product_id          in EVALUATION.product_id%type,
  i_evaluation_scale_id in EVALUATION.evaluation_scale_id%type
) IS
  err_num number;
  err_msg varchar2(255);
  my_old_evaluation_scale_id EVALUATION.evaluation_scale_id%type;
  my_old_evaluation_timestamp EVALUATION.evaluation_timestamp%type;
BEGIN

  select
	evaluation_scale_id,
	evaluation_timestamp
  into
	my_old_evaluation_scale_id,
	my_old_evaluation_timestamp
  from
    EVALUATION
  where
    user_id = i_user_id and
    product_id = i_product_id;

  if (my_old_evaluation_scale_id is null) then
     raise no_data_found;
  elsif (i_evaluation_scale_id != 0) then
  -- we don't want to overwrite an evaluation with 0 over something else
    insert into EVALUATION_HISTORY (
      user_id,
      product_id,
      evaluation_scale_id,
      old_evaluation_timestamp,
      new_evaluation_timestamp
    ) values (
      i_user_id,
      i_product_id,
      my_old_evaluation_scale_id,
      my_old_evaluation_timestamp,
      sysdate
    );

    update EVALUATION set
      evaluation_scale_id = i_evaluation_scale_id,
      evaluation_timestamp = sysdate
    where
      user_id = i_user_id and
      product_id = i_product_id;
    end if;

EXCEPTION

  when no_data_found then

    insert into EVALUATION (
      user_id,
      product_id,
      evaluation_scale_id,
      evaluation_timestamp
    ) values (
      i_user_id,
      i_product_id,
      i_evaluation_scale_id,
      sysdate
    );

/*
  insert into EVALUATION (
    user_id,
	product_id,
	evaluation_scale_id,
	evaluation_timestamp
  ) values (
    i_user_id,
	i_product_id,
	i_evaluation_scale_id,
	sysdate
  );

EXCEPTION

  when dup_val_on_index then

    insert into EVALUATION_HISTORY (
      user_id,
      product_id,
      evaluation_scale_id,
      old_evaluation_timestamp,
      new_evaluation_timestamp
    ) (
      select
        user_id,
        product_id,
        evaluation_scale_id,
        evaluation_timestamp,
        sysdate
      from
        EVALUATION
      where
        user_id = i_user_id and
        product_id = i_product_id
    );

    update EVALUATION set
      evaluation_scale_id = i_evaluation_scale_id,
      evaluation_timestamp = sysdate
    where
      user_id = i_user_id and
      product_id = i_product_id;
*/
 END sp_insert_evals;

-- ==========================================================================================
-- Selecting product evaluations by users

PROCEDURE sp_sel_prodeval_by_user (
  i_user_id           in  EVALUATION.user_id%type,
  o_evaluation_cursor out evaluation_cursor_type,
  o_error_code        out number
) IS
BEGIN

  o_error_code := 0;

  open o_evaluation_cursor for
  select
    product_id,
    evaluation_scale_id
  from
    evaluation
  where
    user_id = i_user_id;

EXCEPTION

  when others then
    o_error_code := 6553; -- general errors when selcting product evaluations by users
    close o_evaluation_cursor;

END sp_sel_prodeval_by_user;


-- ==========================================================================================
-- Selecting user data from USER_DATA_STAT from a random USER_ID (1st time) or no evaluations b/w 1 and 6 --

PROCEDURE sp_sel_user_data (
  i_user_id           in  USER_ID.user_id%type,
  o_evaluation_cursor out evaluation_cursor_type,
  o_error_code        out number
) IS
  v_u_id number;
BEGIN

  o_error_code := 0;

  begin

    -- if there are not user_data_stats for i_user_id, then v_u_id = [random] --
    select
      user_id
    into
      v_u_id
    from
      USER_DATA_STAT
    where
      not exists (
        select 'x'
        from
          user_data_stat
        where
          user_id = i_user_id
      ) and
      rownum = 1;

  exception

    -- if there are user_data_stats for i_user_id, then v_u_id = i_user_id --
    when no_data_found then
      v_u_id := i_user_id;

  end;

  open o_evaluation_cursor for
  select
    product_cluster_id,
    nvl(ni,0) ni,
    nvl(ri,0) ri,
    nvl(ei,0) ei,
    nvl(fi,0) fi,
    nvl(bi,0) bi,
    nvl(eave,0) eave,
    nvl(rave,0) rave
  from
    USER_DATA_STAT
  where
    user_id = v_u_id
  union
  select
    product_cluster_id,
    nvl(min(wgt_tot_prod_in_pc),0),
    0,
    0,
    0,
    0,
    0,
    0
  from
    PC_STAT PS
  where
    not exists (
      select 'x'
      from
        USER_DATA_STAT UDS
      where
        UDS.user_id = v_u_id and
        UDS.product_cluster_id =
        PS.product_cluster_id
    )
  group by
    product_cluster_id;

EXCEPTION

  when others then
    o_error_code := 6601;
    close o_evaluation_cursor;

END sp_sel_user_data;

-- ==========================================================================================
-- Selecting new product data by product cluster by selecting product_id from PROD_PC_STAT where the
-- total number of user evaluations is less than the total number of user evaluations in the PROD_STAT
-- table or the fraction of users that have seen the product is greater than the fraction of users that
-- have seen the product in the PROD_STAT table

PROCEDURE sp_sel_new_prod_data_by_pc1 (
  i_product_cluster_id in  PRODUCT_CLUSTER.product_cluster_id%type,
  i_r                  in  PROD_STAT.frn_user_seen%type,
  i_q                  in  PROD_STAT.tot_user_eval%type,
  i_user_id            in  USER_ID.user_id%type,
  o_evaluation_cursor  out evaluation_cursor_type,
  o_error_code         out number
) IS
BEGIN

  o_error_code := 0;

  open o_evaluation_cursor for
  select
    S.product_id,
    S.product_type_id
  from
    PROD_PC_STAT S
  where
    S.product_cluster_id = i_product_cluster_id and
    S.new_addition_ind = 1 and
    (S.tot_user_eval < i_q or S.frn_user_seen > i_r) and
    not exists (
      select 'x'
      from
        EVALUATION E
      where
        E.user_id = i_user_id and
        E.product_id = S.product_id
    );

EXCEPTION

  when others then
    o_error_code  := 6602;
    close o_evaluation_cursor;

END sp_sel_new_prod_data_by_pc1;

-- ==========================================================================================

PROCEDURE sp_sel_new_prod_data_by_pc (
  i_product_cluster_id in  PRODUCT_CLUSTER.product_cluster_id%type,
  i_user_id            in  USER_ID.user_id%type,
  o_evaluation_cursor  out evaluation_cursor_type,
  o_error_code         out number
) IS
BEGIN

  o_error_code := 0;

  open o_evaluation_cursor for
  select
    S.product_id,
    S.product_type_id
  from
    PROD_PC_STAT S
  where
    S.product_cluster_id = i_product_cluster_id and
    S.new_addition_ind = 1 and
    not exists (
      select 'x'
      from
        EVALUATION E
      where
        E.user_id = i_user_id and
        E.product_id = S.product_id
    );

EXCEPTION

  when others then
    o_error_code  := 6602;
    close o_evaluation_cursor;

END sp_sel_new_prod_data_by_pc;

-- ==========================================================================================
-- Select new product data by product cluster by selecting product_id from PROD_PC_STAT
-- where the total number of user evaluations is greater than the number of user evaluations
-- in the PROD_STAT table and the fraction of users that have seen the product is
-- less than the fraction of users seen the product in the PROD_STAT table

PROCEDURE sp_sel_new_prod_data_by_pc2 (
  i_product_cluster_id in  PRODUCT_CLUSTER.product_cluster_id%type,
  i_r                  in  PROD_STAT.frn_user_seen%type,
  i_q                  in  PROD_STAT.tot_user_eval%type,
  i_user_id            in  USER_ID.user_id%type,
  o_evaluation_cursor  out evaluation_cursor_type,
  o_error_code         out number
) IS
BEGIN

  o_error_code := 0;

  open o_evaluation_cursor for
  select
    S.product_id,
    S.product_type_id
  from
    PROD_PC_STAT S
  where
    S.product_cluster_id = i_product_cluster_id and
    S.new_addition_ind = 1 and
    S.tot_user_eval > i_q and
    S.frn_user_seen < i_r and
    not exists (
      select 'x'
      from
        EVALUATION E
      where
        E.user_id = i_user_id and
        E.product_id = S.product_id
    );


EXCEPTION

  when others then
    o_error_code  := 6603;
    close o_evaluation_cursor;

END sp_sel_new_prod_data_by_pc2;

-- ==========================================================================================
-- Select new product data by given users

PROCEDURE sp_sel_new_prod_data_by_user (
  i_user_id           in  USER_ID.user_id%type,
  o_evaluation_cursor out evaluation_cursor_type,
  o_error_code        out number
) IS
BEGIN

  o_error_code	:= 0;

  open o_evaluation_cursor for
  select
    product_cluster_id,
    new_prod_epi as epi,
    new_prod_ni as ni,
    new_prod_ei as ei,
    new_prod_mi as mi
  from
    USER_DATA_STAT
  where
    user_id = i_user_id
  union
  select
    product_cluster_id,
    0,
    nvl(min(wgt_tot_new_prod_in_pc),0),
    0,
    nvl(min(wgt_tot_new_prod_in_pc),0)
  from
    PC_STAT PS
  where
    not exists (
    select 'x'
    from
      USER_DATA_STAT UDS
    where
      UDS.user_id = i_user_id and
      UDS.product_cluster_id = PS.product_cluster_id)
  group by product_cluster_id;

EXCEPTION

  when others then
    o_error_code  := 6604;
    close o_evaluation_cursor;

END sp_sel_new_prod_data_by_user;

-- ==========================================================================================
-- Selecting old product data by product cluster where its data already exists in sp_SEL_NEW_PROD_DATA_BY_PC1

PROCEDURE sp_sel_old_prod_data_by_pc1 (
  i_product_cluster_id in  PRODUCT_CLUSTER.product_cluster_id%type,
  i_r                  in  PROD_STAT.frn_user_seen%type,
  i_q                  in  PROD_STAT.tot_user_eval%type,
  i_omega              in  PROD_STAT.tot_user_seen%type,
  i_delta              in  PROD_STAT.tot_user_seen%type,
  i_user_id            in  USER_ID.user_id%type,
  o_evaluation_cursor  out evaluation_cursor_type,
  o_error_code         out number
) IS
BEGIN

  o_error_code	:= 0;

  open o_evaluation_cursor for
  select
    S.product_id,
    S.product_type_id
  from
    PROD_PC_STAT S
  where
    S.product_cluster_id = i_product_cluster_id and
    S.new_addition_ind = 0 and
    (S.tot_user_eval < i_q or s.frn_user_seen > i_r) and
    S.tot_user_seen < i_omega and
    S.tot_user_seen > i_delta and
    not exists (
      select 'x'
      from
        EVALUATION E
      where
        E.user_id = i_user_id and
        E.product_id = S.product_id
    );

EXCEPTION

  when others then
     o_error_code  := 6605;
     close o_evaluation_cursor;

END sp_sel_old_prod_data_by_pc1;

-- ==========================================================================================

PROCEDURE sp_sel_old_prod_data_by_pc (
  i_product_cluster_id in  PRODUCT_CLUSTER.product_cluster_id%type,
  i_user_id            in  USER_ID.user_id%type,
  o_evaluation_cursor  out evaluation_cursor_type,
  o_error_code         out number
) IS
BEGIN

  o_error_code := 0;
  open o_evaluation_cursor for

  select
    S.product_id,
    S.product_type_id
  from
    PROD_PC_STAT S
  where
    S.product_cluster_id = i_product_cluster_id and
    S.new_addition_ind = 0 and
    not exists (
      select 'x'
      from
        EVALUATION E
      where
        E.user_id = i_user_id and
        E.product_id = s.product_id
    );

EXCEPTION

  when others then
    o_error_code  := 6605;
    close o_evaluation_cursor;

END sp_sel_old_prod_data_by_pc;

-- ==========================================================================================
-- Selecting old product data by product cluster where its data already exists in sp_SEL_NEW_PROD_DATA_BY_PC2

PROCEDURE sp_sel_old_prod_data_by_pc2 (
  i_product_cluster_id in  PRODUCT_CLUSTER.product_cluster_id%type,
  i_r                  in  PROD_STAT.frn_user_seen%type,
  i_q                  in  PROD_STAT.tot_user_eval%type,
  i_omega              in  PROD_STAT.tot_user_seen%type,
  i_delta              in  PROD_STAT.tot_user_seen%type,
  i_user_id            in  USER_ID.user_id%type,
  o_evaluation_cursor  out evaluation_cursor_type,
  o_error_code         out number
) IS
BEGIN

  o_error_code := 0;

  open o_evaluation_cursor for
  select
    S.product_id,
    S.product_type_id
  from
    PROD_PC_STAT S
  where
    S.product_cluster_id = i_product_cluster_id and
    S.new_addition_ind = 0 and
    S.tot_user_eval > i_q and
    S.frn_user_seen < i_r and
    (S.tot_user_seen < i_omega and s.tot_user_seen > i_delta) and
    not exists (
      select 'x'
      from
        EVALUATION E
      where
        E.user_id = i_user_id and
        E.product_id = S.product_id
    );

EXCEPTION

  when others then
     o_error_code  := 6606;
     close o_evaluation_cursor;

END sp_sel_old_prod_data_by_pc2;

-- ==========================================================================================
-- Selecting old product data by given users

PROCEDURE sp_sel_old_prod_data_by_user (
  i_user_id           in  USER_ID.user_id%type,
  o_evaluation_cursor out evaluation_cursor_type,
  o_error_code        out number
) IS
BEGIN

  o_error_code := 0;

  open o_evaluation_cursor for
  select
    product_cluster_id,
    old_prod_ome_epi as epi,
    old_prod_ome_ni as ni,
    old_prod_ome_ei as ei,
    old_prod_ome_mi as mi
  from
    USER_DATA_STAT
  where
    user_id = i_user_id
  union
  select
    product_cluster_id,
    0,
    nvl(min(ome_wgt_tot_old_prod_in_pc),0),
    0,
    nvl(min(tot_old_prod_in_pc),0)
  from
    PC_STAT PS
  where
    not exists (
      select 'x'
      from
        USER_DATA_STAT UDS
      where
        UDS.user_id = i_user_id and
        UDS.product_cluster_id = PS.product_cluster_id
    )
  group by
    product_cluster_id	;

EXCEPTION

  when others then
    o_error_code  := 6607;
    close o_evaluation_cursor;

END sp_sel_old_prod_data_by_user;

-- ==========================================================================================
-- Selecting product by user data, ordering the result by its average in a descending order
-- MSW 101800, NVL(MIN(),0) was added to return all products in order when user_id of "0" is an input

PROCEDURE sp_sel_prod_by_user_data (
  i_product_cluster_id in  PROD_UC_STAT.product_cluster_id%type,
  i_user_id            in  USER_ID.user_id%type,
  o_evaluation_cursor  out evaluation_cursor_type,
  o_error_code         out number
) IS
  v_U_ID number;
BEGIN

  o_error_code := 0;

  begin

    -- if there are not user_data_stats for i_user_id, then v_u_id = [random] --
    select
      user_id
    into
      v_u_id
    from
      USER_DATA_STAT
    where
      not exists (
        select 'x'
        from
          USER_DATA_STAT
        where
          user_id = i_user_id
      ) and
      rownum = 1;

  exception

    -- if there are user_data_stats for i_user_id, then v_u_id = i_user_id --
    when no_data_found then
      v_u_id := i_user_id;

  end;

  open o_evaluation_cursor for
  select
    s.product_id,
    s.product_type_id
  from
    PROD_PC_STAT S
  where
    S.product_cluster_id = i_product_cluster_id and
    S.frn_user_seen >= (
      select
        max(frn_seen_by_pc_para)
      from
        EVAL_PARAMETER
      where
        tot_eval_by_user_pc_para > (
          select
            nvl(tot_prod_eval,0) -- [RC] 2001/04/25: was "nvl(min(tot_prod_eval),0)"
          from
            USER_PC_STAT
          where
            user_id = v_u_id and
            product_cluster_id = i_product_cluster_id
        )
    ) and
    not exists (
      select
        'x'
      from
        EVALUATION
      where
        user_id = i_user_id and
        product_id = S.product_id
    )
  order by
    avg_v desc;

EXCEPTION

  when others then
    o_error_code  := 6608;
    close o_evaluation_cursor;

END sp_sel_prod_by_user_data;

-- ==========================================================================================
-- Counting the number of times that a product has been seen by users

PROCEDURE sp_cnt_prod_seen_by_user (
  i_user_id       in  evaluation.user_id%type,
  o_tot_prod_seen out number,
  o_error_code    out number
) IS
BEGIN

  o_error_code	:= 0;

  select
    count(evaluation_scale_id)
  into
    o_tot_prod_seen
  from
    EVALUATION
  where
    user_id = i_user_id and
    evaluation_scale_id >= 1;

EXCEPTION

  when others then
    o_error_code  := 6609;

END sp_cnt_prod_seen_by_user;

-- ==========================================================================================

PROCEDURE getUserStats (
  i_user_id       in  evaluation.user_id%type,
  o_tot_prod_eval out number,
  o_tot_prod_seen out number,
  o_frn_prod_seen out number,
  o_tot_core_prod_eval out number,
  o_tot_selectable_prod_eval out number,
  o_error_code    out number
) IS
BEGIN

  o_error_code	:= 0;

  select
    tot_prod_eval,
    tot_prod_seen,
    frn_prod_seen,
    tot_core_prod_eval,
    tot_selectable_prod_eval
  into
    o_tot_prod_eval,
    o_tot_prod_seen,
    o_frn_prod_seen,
    o_tot_core_prod_eval,
    o_tot_selectable_prod_eval
  from
    USER_STAT
  where
    user_id = i_user_id;

EXCEPTION

  when others then
    o_error_code  := 6218;

END getUserStats;

-- ==========================================================================================

PROCEDURE getEveryoneStats (
  o_tot_prod_eval out number,
  o_tot_prod_seen out number,
  o_frn_prod_seen out number,
  o_tot_core_prods out number,
  o_tot_selectable_prods out number,
  o_error_code    out number
) IS
BEGIN

  o_error_code	:= 0;

  select
    count(distinct product_id)
  into
    o_tot_prod_eval
  from
    EVALUATION;

  select
    count(distinct product_id)
  into
    o_tot_prod_seen
  from
    RATING;

  o_frn_prod_seen := o_tot_prod_seen / o_tot_prod_eval;

  select
    count(*)
  into
    o_tot_core_prods
  from
    CORE_PROD;

  select
    count(*)
  into
    o_tot_selectable_prods
  from
    SELECTABLE_PROD;

EXCEPTION

  when others then
    o_error_code  := 6219;

END getEveryoneStats;

END pkg_alkindi_evaluation;
/
